 /*
   *  Object %name    : %CRYS_ECDSA_Verify.c
   *  State           :  %state%
   *  Creation date   :  16.01.2007
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This module contains the functions used to verifying the ECDSA signature.
   *
   *  \version CRYS_ECDSA_Verify.c#1:csrc:1
   *  \author R.Levin
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/

/* .............. CRYS level includes ................. */
#include "DX_VOS_Mem.h"
#include "PLAT_SystemDep.h"
#include "CRYS_ECPKI_error.h"
#include "CRYS.h"
#include "dx_hw_defs.h"
#include "crys_host_op_code.h"


/* .............. LLF level includes ................. */

/* canceling the lint warning:
   Warning 548: else expected */
/*lint --e{548} */

/* canceling the lint warning:
Info 801: Use of goto is deprecated */
/*lint --e{801} */

/************************ Defines *****************************************/

/************************ Enums *******************************************/

/************************ Typedefs ****************************************/

/************************ Global Data *************************************/

/************* Private function prototype *********************************/

/************************ Public Functions ********************************/

/**************************************************************************
 *	              CRYS_ECDSA_VerifyInit  function
 **************************************************************************/
/**
   @brief  Prepares a context that is used by the Update and Finish functions
           but does not perform elliptic curve cryptographic processing

		    The function:
			- Receives and decrypts user data (working context). 
			- Checks input parameters of  ECDSA Vrifying primitive.
			- Calls hash init function.
			- Initializes variables and structures for calling next functions.
			- Encrypts and releases working context.
			
			NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
			is not recommended!

   @param[in,out] VerifyUserContext_ptr - A pointer to the user buffer for verifying database.
   @param[in] SignerPublKey_ptr - A pointer to a Signer public key structure.
   @param[in] HashMode - The enumerator variable defines the hash function to be used.
   
   @return <b>CRYSError_t</b>: <br> 
			 CRYS_OK<br>
                         CRYS_ECDSA_VERIFY_INVALID_USER_CONTEXT_PTR_ERROR <br>
			 CRYS_ECDSA_VERIFY_INVALID_SIGNER_PUBL_KEY_PTR_ERROR <br>
                         CRYS_ECDSA_VERIFY_SIGNER_PUBL_KEY_VALIDATION_TAG_ERROR <br>
                         CRYS_ECDSA_VERIFY_INVALID_DOMAIN_ID_ERROR <br>
			 CRYS_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR <br>
**/
CEXPORT_C CRYSError_t CRYS_ECDSA_VerifyInit( 
	                               CRYS_ECDSA_VerifyUserContext_t  *VerifyUserContext_ptr, /*in/out*/
								   CRYS_ECPKI_UserPublKey_t    	   *SignerPublKey_ptr,     /*in*/
								   CRYS_ECPKI_HASH_OpMode_t         HashMode               /*in*/ )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_VERIFY_INIT_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 VerifyUserContext_ptr,
								 SignerPublKey_ptr,
								 HashMode );

}/* _DX_ECDSA_VerifyInit */



/**************************************************************************
 *	              CRYS_ECDSA_VerifyUpdate function
 **************************************************************************/
/**
   @brief  Performs a hash  operation on data allocated by the user 
           before finally verifying its signature.
  
   	   In case user divides signing data by block, he must call the Update function 
   	   continuously a number of times until processing of the entire data block is complete.
   	   
       NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
             is not recommended!

   @param[in,out] VerifyUserContext_ptr A pointer to the user buffer for verifying database.
   @param[in] 	  MessageDataIn_ptr   Message data for calculating Hash.   	
   @param[in]     DataInSize            The size of the message data block, in bytes.
   
   @return <b>CRYSError_t</b>: <br> 
			 CRYS_OK<br>
                         CRYS_ECDSA_VERIFY_INVALID_USER_CONTEXT_PTR_ERROR <br>
			 CRYS_ECDSA_VERIFY_USER_CONTEXT_VALIDATION_TAG_ERROR <br>
			 CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_PTR_ERROR <br>
			 CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_SIZE_ERROR <br>
			 CRYS_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR <br>
 **/
CEXPORT_C CRYSError_t CRYS_ECDSA_VerifyUpdate( 
	                                 CRYS_ECDSA_VerifyUserContext_t *VerifyUserContext_ptr, /*in/out*/
								     DxUint8_t                      *MessageDataIn_ptr,     /* in */ 
							  	     DxUint32_t                      DataInSize             /* in */ )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_VERIFY_UPDATE_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 VerifyUserContext_ptr,
								 MessageDataIn_ptr,
								 DataInSize );

}/* CRYS_ECDSA_VerifyUpdate */

				  

/**************************************************************************
 *	              CRYS_ECDSA_VerifyFinish function
 **************************************************************************/
/**
   @brief  Performs initialization of variables and structures, 
           calls the hash function for the last block of data (if necessary),  
   		   than calls LLF_ECDSA_VerifyCalcCall function for verifying signature  
		   according to EC DSA algorithm.

       NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
			 is not recommended!	   

   @param[in] VerifyUserContext_ptr - A pointer to the user buffer for verifying the database.
   @param[in] SignatureIn_ptr       - A pointer to a buffer for the signature to be compared
   @param[in] SignatureSizeBytes    - The size of a user passed signature (must be 2*OrderSizeInBytes).

   @return <b>CRYSError_t</b>: <br>
              CRYS_OK <br>
			  CRYS_ECDSA_VERIFY_INVALID_USER_CONTEXT_PTR_ERROR <br>
			  CRYS_ECDSA_VERIFY_USER_CONTEXT_VALIDATION_TAG_ERROR <br>
			  CRYS_ECDSA_VERIFY_INVALID_SIGNATURE_IN_PTR_ERROR <br>
			  CRYS_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR <br>
			  CRYS_ECDSA_VERIFY_INVALID_SIGNATURE_SIZE_ERROR <br>			  
			  CRYS_ECDSA_VERIFY_INVALID_DOMAIN_ID_ERROR <br>
			  CRYS_ECDSA_VERIFY_INCONSISTENT_VERIFY_ERROR <br>
**/
CEXPORT_C CRYSError_t CRYS_ECDSA_VerifyFinish( 
	                         CRYS_ECDSA_VerifyUserContext_t *VerifyUserContext_ptr, /*in*/ 
							 DxUint8_t                      *SignatureIn_ptr,       /*in*/
							 DxUint32_t                      SignatureSizeBytes      /*in*/  )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_VERIFY_FINISH_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 VerifyUserContext_ptr,
								 SignatureIn_ptr,
								 SignatureSizeBytes );

}/* End DX_ECDSA_VerifyFinish */


/**************************************************************************
 *	              CRYS_ECDSA_Verify integrated function
 **************************************************************************/
/**
   @brief  Performs all ECDSA verifying operations simultaneously. 
		   
	   This function simply calls the Init, Update and Finish functions continuously.  
   	   This function's prototype is similar to the prototypes of the called functions 
   	   and includes all of their input and output arguments.
   	   
       NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
			 is not recommended!
   	   

   @param[in]  VerifyUserContext_ptr - A pointer to the user buffer for verifying database.
   @param[in]  UserPublKey_ptr       - A pointer to a user public key structure.
   @param[in]  HashMode              - The enumerator variable defines the hash function to be used.
   @param[in]  MessageDataIn_ptr     - Message data for calculating hash.   			
   @param[in]  MessageSizeInBytes    - Size of block of message data in bytes.       
   @param[in]  SignatureIn_ptr       - A pointer to a buffer for output of signature. 
   @param[in]  SignatureSizeBytes    - Size of signature, in bytes (must be 2*OrderSizeInBytes).
                                       
   @return <b>CRYSError_t</b>: <br> 
              CRYS_OK <br>
              CRYS_ECDSA_VERIFY_INVALID_USER_CONTEXT_PTR_ERROR <br> 
              CRYS_ECDSA_VERIFY_USER_CONTEXT_VALIDATION_TAG_ERROR <br>
              CRYS_ECDSA_VERIFY_INVALID_DOMAIN_ID_ERROR <br>
              CRYS_ECDSA_VERIFY_INVALID_SIGNER_PUBL_KEY_PTR_ERROR <br>
              CRYS_ECDSA_VERIFY_SIGNER_PUBL_KEY_VALIDATION_TAG_ERROR <br>
              CRYS_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR <br>                
              CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_PTR_ERROR <br>
              CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_SIZE_ERROR <br>						                        
              CRYS_ECDSA_VERIFY_INVALID_SIGNATURE_IN_PTR_ERROR <br>						 
              CRYS_ECDSA_VERIFY_INVALID_SIGNATURE_SIZE_ERROR <br>
			  CRYS_ECDSA_VERIFY_INCONSISTENT_VERIFY_ERROR <br>
**/
CEXPORT_C CRYSError_t CRYS_ECDSA_Verify ( 
	                    CRYS_ECDSA_VerifyUserContext_t *VerifyUserContext_ptr, /*in/out*/
						CRYS_ECPKI_UserPublKey_t       *UserPublKey_ptr,        /*in*/
						CRYS_ECPKI_HASH_OpMode_t        HashMode,               /*in*/
						DxUint8_t                      *SignatureIn_ptr,        /*in*/
						DxUint32_t                      SignatureSizeBytes,     /*in*/
						DxUint8_t                      *MessageDataIn_ptr,      /*in*/ 
						DxUint32_t                      MessageSizeInBytes      /*in*/ )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_VERIFY_OP_CODE, 7,
	   							 /* PARAMETERS: */
								 VerifyUserContext_ptr,
								 UserPublKey_ptr,
								 HashMode,
								 SignatureIn_ptr,
								 SignatureSizeBytes,
								 MessageDataIn_ptr,
								 MessageSizeInBytes );

}/* END OF CRYS_ECDSA_Verify */

